/*
 * linux/drivers/video/backlight/pwm_bl.c
 *
 * simple PWM based backlight control, board code has to setup
 * 1) pin configuration so PWM waveforms can output
 * 2) platform_data being correctly configured
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <linux/err.h>
#include <linux/pwm.h>
#include <linux/pwm_backlight.h>
#include <linux/slab.h>

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */
#include <mach/gpio.h>

#define PROC_LCD_FILE   "lcdBklt"
#define PROCFS_MAX_SIZE     2048

#define MAX_BRIGHTNESS 100 // max value is 100%
#define MIN_BRIGHTNESS 0

#define MAX_DEV_BRIGHTNESS 255 //device MAX value,

struct pwm_bl_data {
        struct pwm_device       *pwm;
        struct device           *dev;
        unsigned int            period;
        int                     (*notify)(struct device *,
                                          int brightness);
};

struct lcdpanel {
        u32 brightness;
        u32 panelType;
        u32 colorDepth;
        u32 pixelHeight;
        u32 pixelWidth;
        u32 mmHeight;
        u32 mmWidth;
        u32 lcdEnabled;
        u32 backlightLevelOnBatteryDef;
        u32 backlightLevelOnExtDef;
};

struct lcdpanel *gLcdPanel;
struct backlight_device *gBlData;
static int pwm_backlight_update_status(struct backlight_device *bl);
static int pwm_backlight_get_brightness(struct backlight_device *bl);

/* 0-255 is passed into this function */
u32 calcDeviceToUI(int brtness)
{
	/* convert 255 to 100, multiplied times 10 to avoid floating point*/
	u32 val;
	if (brtness) //avoid divide by zero
		val = ((brtness*100) / MAX_DEV_BRIGHTNESS); //max is 255 converted to 100%
	else
		val=0;

	return (val);
}
/* 0-100% is passed in and percentage is returned */
u32 calcUiToDevice(int brtness)
{
	/* convert 100 to 255 */
	u32 val;
	if (brtness)  //avoid divide by zero
		val = ((brtness * MAX_DEV_BRIGHTNESS)/100); // max is 100% converted to 255
	else
		val=0;

	return (val);
}


/**
 * The structure keeping information about the /proc file
 *
 */
static struct proc_dir_entry *lcd_file;

/**
 * The buffer (2k) for this module
 *
 */
static char procfs_buffer_lcd[PROCFS_MAX_SIZE];

/**
 * The size of the data held in the buffer
 *
 */
static unsigned long procfs_buffer_size_lcd = 0;

/**
 * This funtion is called when the /proc file is read
 *
 */
static ssize_t procfs_read_lcd(struct file *filep,      /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{

   char * p = procfs_buffer_lcd;
   static int finished = 0;

   /* needed to stop from continuously printing */
   if ( finished == 1 ) { finished=0; return 0; }
   finished = 1;

   p += sprintf ( p, "BRIGHTNESS=%d\n" , gLcdPanel->brightness );
   p += sprintf ( p, "PANEL_TYPE=%d\n",  gLcdPanel->panelType );
   p += sprintf ( p, "COLOR_DEPTH=%d\n", gLcdPanel->colorDepth );
   p += sprintf ( p, "PIXEL_HEIGHT=%d\n", gLcdPanel->pixelHeight );
   p += sprintf ( p, "PIXEL_WIDTH=%d\n", gLcdPanel->pixelWidth );
   p += sprintf ( p, "MILLIMETER_HEIGHT=%d\n", gLcdPanel->mmHeight );
   p += sprintf ( p, "MILLIMETER_WIDTH=%d\n", gLcdPanel->mmWidth );
   p += sprintf ( p, "LCD_ENABLED=%d\n", gLcdPanel->lcdEnabled );
   p += sprintf ( p, "BACKLIGHT_LEVEL_ON_BATTERY_DEF=%d\n", gLcdPanel->backlightLevelOnBatteryDef );
   p += sprintf ( p, "BACKLIGHT_LEVEL_ON_EXT_DEF=%d\n", gLcdPanel->backlightLevelOnExtDef );

   procfs_buffer_size_lcd = p - procfs_buffer_lcd;

   if ( copy_to_user(buffer, procfs_buffer_lcd, procfs_buffer_size_lcd) ) {
              return -EFAULT;
   }

  return procfs_buffer_size_lcd;
}

static ssize_t procfs_write_lcd(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;
	int tempVal=0;

        procfs_buffer_size_lcd = len;
        if (procfs_buffer_size_lcd > PROCFS_MAX_SIZE ) {
                procfs_buffer_size_lcd = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(procfs_buffer_lcd, buffer, procfs_buffer_size_lcd) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "LCD_ENABLED" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &gLcdPanel->lcdEnabled );
        }
        if ( strcmp ( tag, "BRIGHTNESS" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &tempVal );
	  if (tempVal <= MAX_BRIGHTNESS)
	  {
	  gLcdPanel->brightness=tempVal;
	  gBlData->props.brightness=calcUiToDevice(gLcdPanel->brightness);
	  pwm_backlight_update_status(gBlData);
	  }
	  else
	  {
		printk(KERN_ERR "brightness percentage value is out of range\n");
	  }
        }
        if ( strcmp ( tag, "BACKLIGHT_LEVEL_ON_BATTERY_DEF" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &tempVal );
	  if (tempVal <= MAX_BRIGHTNESS)
	  {
	  	gLcdPanel->backlightLevelOnBatteryDef=tempVal;
	  }
	  else
	  {
		printk(KERN_ERR "backlightLevelOnBatteryDef percentage value is out of range\n");
	  }
        }
        if ( strcmp ( tag, "BACKLIGHT_LEVEL_ON_EXT_DEF" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &tempVal );
	  if (tempVal <= MAX_BRIGHTNESS)
	  {
	  	gLcdPanel->backlightLevelOnExtDef=tempVal;
	  }
	  else
	  {
		printk(KERN_ERR "backlightLevelOnExtDef percentage value is out of range\n");
	  }
        }

	return procfs_buffer_size_lcd;

}

static int module_permission_lcd(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int procfs_open_lcd(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int procfs_close_lcd(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_Lcd_File = {
        .read    = procfs_read_lcd,
        .write   = procfs_write_lcd,
        .open    = procfs_open_lcd,
        .release = procfs_close_lcd,
};

static struct inode_operations Inode_Ops_Lcd_File = {
        .permission = module_permission_lcd,    /* check for permissions */
};

static void initIntfcValues(struct platform_pwm_backlight_data *dta) 
{
	//gLcdPanel->brightness= 25;  //RJK, do not set here, set in _probe function
	gLcdPanel->panelType =	1;  //assume WVGA TFT LCD
	gLcdPanel->colorDepth =	dta->color_depth;
	gLcdPanel->pixelHeight =	dta->pixel_height;
	gLcdPanel->pixelWidth = 	dta->pixel_width;
	gLcdPanel->mmHeight =	dta->mm_height;
	gLcdPanel->mmWidth =	dta->mm_width;
	gLcdPanel->lcdEnabled =1;
}

static int pwm_backlight_update_status(struct backlight_device *bl)
{
	struct pwm_bl_data *pb = dev_get_drvdata(&bl->dev);
	int brightness = bl->props.brightness;
	int max = bl->props.max_brightness;

	if (bl->props.power != FB_BLANK_UNBLANK)
		brightness = 0;

	if (bl->props.fb_blank != FB_BLANK_UNBLANK)
		brightness = 0;

	if (pb->notify)
		brightness = pb->notify(pb->dev, brightness);

	if (brightness == 0) {
		pwm_config(pb->pwm, 0, pb->period);
		pwm_disable(pb->pwm);
	} else {
		if (bl->props.invert_brightness)
		{
			brightness = 255 - brightness;
		}
		pwm_config(pb->pwm, brightness * pb->period / max, pb->period);
		pwm_enable(pb->pwm);
	}
	return 0;
}

static int pwm_backlight_get_brightness(struct backlight_device *bl)
{
	return bl->props.brightness;
}

static int pwm_backlight_check_fb(struct backlight_device *bl, struct fb_info *info)
{
	return 0;
}

static const struct backlight_ops pwm_backlight_ops = {
	.update_status	= pwm_backlight_update_status,
	.get_brightness	= pwm_backlight_get_brightness,
	.check_fb = pwm_backlight_check_fb,
};

static int pwm_backlight_probe(struct platform_device *pdev)
{
	struct backlight_properties props;
	struct platform_pwm_backlight_data *data = pdev->dev.platform_data;
	struct backlight_device *bl;
	struct pwm_bl_data *pb;
	int ret;

//	printk(KERN_ERR "pwm_backlight_probe\n");

	if (!data) {
		dev_err(&pdev->dev, "failed to find platform data\n");
		return -EINVAL;
	}

	if (data->init) {
		ret = data->init(&pdev->dev);
		if (ret < 0)
			return ret;
	}

	pb = kzalloc(sizeof(*pb), GFP_KERNEL);
	if (!pb) {
		dev_err(&pdev->dev, "no memory for state\n");
		ret = -ENOMEM;
		goto err_alloc;
	}

	pb->period = data->pwm_period_ns;
	pb->notify = data->notify;
	pb->dev = &pdev->dev;

	pb->pwm = pwm_request(data->pwm_id, "backlight");
	if (IS_ERR(pb->pwm)) {
		dev_err(&pdev->dev, "unable to request PWM for backlight\n");
		ret = PTR_ERR(pb->pwm);
		goto err_pwm;
	} else
		dev_dbg(&pdev->dev, "got pwm for backlight\n");

	memset(&props, 0, sizeof(struct backlight_properties));
	props.max_brightness = data->max_brightness;
	props.invert_brightness = data->invert_brightness;
	bl = backlight_device_register(dev_name(&pdev->dev), &pdev->dev, pb,
				       &pwm_backlight_ops, &props);
	if (IS_ERR(bl)) {
		dev_err(&pdev->dev, "failed to register backlight\n");
		ret = PTR_ERR(bl);
		goto err_bl;
	}

        gLcdPanel = kzalloc(sizeof(*gLcdPanel), GFP_KERNEL);
        if (!gLcdPanel) {
                ret = -ENOMEM;
                goto err_bl;
        }

	bl->props.brightness = data->dft_brightness;
	gBlData=bl; //RJK globalize data for /proc access

	// init brightness global structure here
	// inits to default brightness
	// RJK: need to be read from a file to maintain persistance
	gLcdPanel->brightness = calcDeviceToUI(bl->props.brightness);
	gLcdPanel->backlightLevelOnBatteryDef = gLcdPanel->brightness;
	gLcdPanel->backlightLevelOnExtDef = gLcdPanel->brightness;

	/* update status and data */
	backlight_update_status(bl);
	platform_set_drvdata(pdev, bl);

	/* setup UiHAL interface through /proc files */
        /* allocate memory for global control structure */
	initIntfcValues(data);

	/* create the /proc file */
        lcd_file = create_proc_entry(PROC_LCD_FILE, 0644, NULL);
        /* check if the /proc file was created successfuly */
        if (lcd_file == NULL){
                printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
                       PROC_LCD_FILE);
                return -ENOMEM;
        }
        else
        {
          //lcd_file->owner = THIS_MODULE;
          lcd_file->proc_iops = &Inode_Ops_Lcd_File;
          lcd_file->proc_fops = &File_Ops_Lcd_File;
          lcd_file->mode = S_IFREG | S_IRUGO | S_IWUSR;
          lcd_file->uid = 0;
          lcd_file->gid = 0;
          lcd_file->size = 80;
       }

        return 0;
 
err_bl:
         pwm_free(pb->pwm);
err_pwm:
         kfree(pb);
         kfree(gLcdPanel);
err_alloc:
        if (data->exit)
               data->exit(&pdev->dev);
        return ret;
};


static int pwm_backlight_remove(struct platform_device *pdev)
{
	struct platform_pwm_backlight_data *data = pdev->dev.platform_data;
	struct backlight_device *bl = platform_get_drvdata(pdev);
	struct pwm_bl_data *pb = dev_get_drvdata(&bl->dev);

	backlight_device_unregister(bl);
	pwm_config(pb->pwm, 0, pb->period);
	pwm_disable(pb->pwm);
	pwm_free(pb->pwm);
	kfree(pb);
	if (data->exit)
		data->exit(&pdev->dev);
	return 0;
}

#ifdef CONFIG_PM
static int pwm_backlight_suspend(struct platform_device *pdev,
				 pm_message_t state)
{
	struct backlight_device *bl = platform_get_drvdata(pdev);
	struct pwm_bl_data *pb = dev_get_drvdata(&bl->dev);

printk(KERN_ERR "pwm_backlight_suspend\n");

	if (pb->notify)
		pb->notify(pb->dev, 0);
	pwm_config(pb->pwm, 0, pb->period);
	pwm_disable(pb->pwm);
	return 0;
}

static int pwm_backlight_resume(struct platform_device *pdev)
{
	struct backlight_device *bl = platform_get_drvdata(pdev);
printk(KERN_ERR "pwm_backlight_resume\n");

	backlight_update_status(bl);
	return 0;
}
#else
#define pwm_backlight_suspend	NULL
#define pwm_backlight_resume	NULL
#endif

static struct platform_driver pwm_backlight_driver = {
	.driver		= {
		.name	= "pwm-backlight",
		.owner	= THIS_MODULE,
	},
	.probe		= pwm_backlight_probe,
	.remove		= pwm_backlight_remove,
	.suspend	= pwm_backlight_suspend,
	.resume		= pwm_backlight_resume,
};

static int __init pwm_backlight_init(void)
{
	return platform_driver_register(&pwm_backlight_driver);
}
module_init(pwm_backlight_init);

static void __exit pwm_backlight_exit(void)
{
	platform_driver_unregister(&pwm_backlight_driver);
}
module_exit(pwm_backlight_exit);

MODULE_DESCRIPTION("PWM based Backlight Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:pwm-backlight");

